/* sccsid[] = "@(#) $Id: //bas/46C/src/proj/crfc/crfcfunc.cpp#2 $ SAP" */   
//
// SAP RFC Classes C++ library.
// Copyright (C) 1996 SAP America, Inc.
// All rights reserved. 

///////////////////////////////////////////////////////////////////////////////////////
//  File crfcfunc.cpp

#if defined(SAPonRS6000)
#pragma priority(90)
#endif

#include "sapitab.h"
#include "saprfc.h"
    
#include "crfcstrc.h"        
#include "crfcfunc.h"
#include "crfcconn.h"
#include "crfcsimp.h"
#include "crfctabl.h"
#include "crfcgfit.h"
  
//initialize transaction ID 
RFC_TID   CRfcFunction::rfcTID="" ;

                                                              
///////////////////////////////////////////////////////////////////////////////////////
//    CRfcFunction definition                                


CRfcFunction::CRfcFunction (const CRfcConnection* pConnection, CSTR strFuncName)
				  :m_pConnection(pConnection),
				  m_rstrFuncName(strFuncName, FUNCTIONNAME_LEN),
                  m_Exports(), 
                  m_Imports(), 
                  m_Tables()
{
	m_rstrFuncName.SetCapitalize(TRUE);

    m_pImpStructs = NULL;
    m_pExpStructs = NULL; 
    m_pTabStructs = NULL;

    m_bImportsInvalid = TRUE;
    m_bExportsInvalid = TRUE;
    m_bTablesInvalid  = TRUE;
}


void CRfcFunction::FreeCRfcFunction(void)
{
	if( m_pImpStructs != NULL )
	{
		delete [] m_pImpStructs;
		m_pImpStructs = NULL;
	}
    if( m_pExpStructs != NULL ) 
	{
		delete [] m_pExpStructs;
		m_pExpStructs = NULL;
	}
    if( m_pTabStructs != NULL ) 
	{
		delete [] m_pTabStructs;
		m_pTabStructs = NULL;
	}
}


CRfcFunction::~CRfcFunction()
{
	FreeCRfcFunction();
}



void CRfcFunction::CopyCRfcFunction(const CRfcFunction& sourceFunc)
{
    m_pConnection = sourceFunc.m_pConnection;
    m_rstrFuncName = sourceFunc.m_rstrFuncName;

	int i;

	m_ProgramNames = sourceFunc.m_ProgramNames;

	m_Exports.RemoveAll();
	for (i = 0; i < sourceFunc.m_Exports.GetSize(); i++)
	{
		CRfcSimpleParam * pTempParam = (CRfcSimpleParam*)sourceFunc.m_Exports[i];
		CRfcImpExpParam *pExpParam = new CRfcSimpleParam(*pTempParam);
		m_Exports.AppendElement(*pExpParam);
	}

	m_Imports.RemoveAll();
	for (i = 0; i < sourceFunc.m_Imports.GetSize(); i++)
	{
		CRfcSimpleParam * pTempParam = (CRfcSimpleParam*)sourceFunc.m_Imports[i];
		CRfcImpExpParam *pImpParam = new CRfcSimpleParam(*pTempParam);
		m_Imports.AppendElement(*pImpParam);
	}

	m_Tables.RemoveAll();
	for (i = 0; i < sourceFunc.m_Tables.GetSize(); i++)
	{
		CRfcTableParam *pTabParam = new CRfcTableParam(*(sourceFunc.m_Tables[i]));
		m_Tables.AppendElement(*pTabParam);
	}
	m_pImpStructs = sourceFunc.m_pImpStructs;
	m_pExpStructs = sourceFunc.m_pExpStructs;
	m_pTabStructs = sourceFunc.m_pTabStructs;
	m_bImportsInvalid = sourceFunc.m_bImportsInvalid;
	m_bExportsInvalid = sourceFunc.m_bExportsInvalid;
	m_bTablesInvalid = sourceFunc.m_bTablesInvalid;
}



//  No copy constructor will be used directly for this ABSTRACT class, 
//  the derived class will use the copy constructor.  

CRfcFunction::CRfcFunction(const CRfcFunction& sourceFunc)
{
	CopyCRfcFunction(sourceFunc);
}

           
//  This is an abstract class, you should noy be able to do assignment operation                                                  
// Use clone() member function for deep copy.
CRfcFunction& CRfcFunction::operator= (const CRfcFunction& sourceFunc )
{
    if( this== &sourceFunc ) return *this;

	FreeCRfcFunction();

	CopyCRfcFunction(sourceFunc);

    return *this;
}

void CRfcFunction::AddImportParam(CRfcImpExpParam& param)
{ 
    //disallow duplicate parameters
    assert(m_Imports[param.GetParamName()]==NULL);

	if (m_Imports[param.GetParamName()]==NULL)
	{
		m_Imports.AppendElement(param);
		param.SetParentFunction(this);
		m_bImportsInvalid = TRUE;
	}
}

void CRfcFunction::AddExportParam(CRfcImpExpParam& param)
{ 
    //disallow duplicate parameters
    assert(m_Exports[param.GetParamName()]==NULL);

	if (m_Exports[param.GetParamName()]==NULL)
	{
		m_Exports.AppendElement(param);
		param.SetParentFunction(this);
		m_bExportsInvalid = TRUE;
	}
}

void CRfcFunction::AddTableParam(CRfcTableParam&  table)
{ 
    //disallow duplicate parameters
    assert(m_Tables[table.GetParamName()]==NULL);

	if (m_Tables[table.GetParamName()]==NULL)
	{
		m_Tables.AppendElement(table);
		table.SetParentFunction(this);
		m_bTablesInvalid = TRUE;
	}
}



//   Replace import parameter object for the given parameter name of the old parameter object
//   Return TRUE if successfull, FALSE if not.

BOOL CRfcFunction::ReplaceImportParam (CRfcImpExpParam * newParam)  
{
	if (newParam && strcmp(newParam->GetParamName(),""))
	{
		CRfcImpExpParam * oldParam = m_Imports.ReplaceElement((CRfcString)newParam->GetParamName(), newParam);
		if (oldParam)
		{
			if (oldParam->IsAutoCreated())
				delete oldParam;
			return TRUE;
		}
		else
			return FALSE;
	}
	else
		return FALSE;
}



//   Replace export parameter object for the given parameter name of the old parameter object
//   Return TRUE if successfull, FALSE if not.

BOOL CRfcFunction::ReplaceExportParam (CRfcImpExpParam * newParam)  
{
	if (newParam && strcmp(newParam->GetParamName(),""))
	{
		CRfcImpExpParam * oldParam = m_Exports.ReplaceElement((CRfcString)newParam->GetParamName(), newParam);
		if (oldParam)
		{
			if (oldParam->IsAutoCreated())
				delete oldParam;
			return TRUE;
		}
		else
			return FALSE;
	}
	else
		return FALSE;
}


//   Replace table parameter object for the given parameter name of the old parameter object
//   Return TRUE if successfull, FALSE if not.

BOOL CRfcFunction::ReplaceTableParam (CRfcTableParam * newParam)
{
	if (newParam && strcmp(newParam->GetParamName(), ""))
	{
		CRfcTableParam * oldParam = m_Tables.ReplaceElement((CRfcString)newParam->GetParamName(), newParam);
		if (oldParam)
		{
			if (oldParam->IsAutoCreated())
				delete oldParam;
			return TRUE;
		}
		else
			return FALSE;
	}
	else
		return FALSE;
}



//  Remove import parameter object for the given name of the parameter
//   Return TRUE if successfull, FALSE if not.

BOOL CRfcFunction::RemoveImportParam (CSTR strParamName)
{
		CRfcImpExpParam * oldParam = m_Imports.RemoveElement(strParamName);
		if (oldParam)
		{
			if (oldParam->IsAutoCreated())
				delete oldParam;
			return TRUE;
		}
		else
			return FALSE;	
}


//  Remove import parameter object for the given index of the parameter
//   Return TRUE if successfull, FALSE if not.

BOOL CRfcFunction::RemoveImportParam (int nIndex)
{
		CRfcImpExpParam * oldParam = m_Imports.RemoveElement(nIndex);
		if (oldParam)
		{
			if (oldParam->IsAutoCreated())
				delete oldParam;
			return TRUE;
		}
		else
			return FALSE;	
}



//  Remove export parameter object for the given name of the parameter
//   Return TRUE if successfull, FALSE if not.

BOOL CRfcFunction::RemoveExportParam (CSTR strParamName)
{
	CRfcImpExpParam * oldParam = m_Exports.RemoveElement(strParamName);
	if (oldParam)
	{
		if (oldParam->IsAutoCreated())
			delete oldParam;
		return TRUE;
	}
	else
		return FALSE;		
}


//  Remove export parameter object for the given index of the parameter
//   Return TRUE if successfull, FALSE if not.

BOOL CRfcFunction::RemoveExportParam (int nIndex)
{
		CRfcImpExpParam * oldParam = m_Exports.RemoveElement(nIndex);
		if (oldParam)
		{
			if (oldParam->IsAutoCreated())
				delete oldParam;
			return TRUE;
		}
		else
			return FALSE;	
}



//  Remove table parameter object for the given name of the parameter
//   Return TRUE if successfull, FALSE if not.

BOOL CRfcFunction::RemoveTableParam (CSTR strParamName)
{
	CRfcTableParam * oldParam = m_Tables.RemoveElement(strParamName);
	if (oldParam)
	{
		if (oldParam->IsAutoCreated())
			delete oldParam;
		return TRUE;
	}
	else
		return FALSE;		
}


//  Remove table parameter object for the given index of the parameter
//   Return TRUE if successfull, FALSE if not.

BOOL CRfcFunction::RemoveTableParam (int nIndex)
{
	CRfcTableParam * oldParam = m_Tables.RemoveElement(nIndex);
	if (oldParam)
	{
		if (oldParam->IsAutoCreated())
			delete oldParam;
		return TRUE;
	}
	else
		return FALSE;		
}


RFC_RC CRfcFunction::Listen (void) // Listen for communication status
{
	RFC_HANDLE RfcHandle = GetConnectionRfcHandle();
	if (RfcHandle == RFC_HANDLE_NULL)
		throw INVALID_RFC_HANDLE;

    return ::RfcListen(RfcHandle);

//    if( rc != RFC_OK )
//    {
//        CRFC_ERROR_INFO errorInfo;
//        ::RfcLastErrorEx(&errorInfo);
	//        ::RfcLastError(&errorInfo);
//        throw errorInfo;
//    }
}

RFC_HANDLE CRfcFunction::GetConnectionRfcHandle(void) const 
{ 
	if (m_pConnection) 
		return m_pConnection->GetHandle (); 
	else	
		return RFC_HANDLE_NULL; 
}


void CRfcFunction::SetUpImports(BOOL bPassAllParams)
{
    int nImpCount = 0;
 
    // Loop to see if all parameters are to be passed to the ABAP function module
    //  only when the CRfcImpExpParam::PassParamToR3() returns TRUE.  Count the
    //  number of params that will actually be passed to R/3.
    // Of course, if the flag bPassAllParams is set, then there is no need to 
    //  check and count.
    if(bPassAllParams)
        nImpCount = m_Imports.GetSize();
    else
    {
        for( int i=0; i < m_Imports.GetSize(); i++ )
            if(m_Imports[i]->PassParamToR3())
                ++nImpCount;
    }

    if( m_bImportsInvalid )
    {    
        if( m_pImpStructs!=NULL ) delete[] m_pImpStructs;  

        // Allocate memory for RFC_PARAMETER structures for parameter passing
        //  to R/3 ABAP function module.
        m_pImpStructs 
           = (RFC_PARAMETER *)new BYTE[sizeof(RFC_PARAMETER)*(nImpCount+1)];
        if( m_pImpStructs == NULL )
        {
            RfcClassTrace.TraceTime();
            RfcClassTrace.Trace(CRFCFUNC_SETUPIMPORTS);
            RfcClassTrace.Trace(MEMORY);
            throw MEMORY;
        }

        m_bImportsInvalid=FALSE;
    }

    // Loop to fill list of RFC_PARAMETER structures.  Note that if bPassAllParams
    //  is set, then every parameter will be passed.
    for(int i=0, j=0; i < m_Imports.GetSize(); i++ )
    {
        if(bPassAllParams || m_Imports[i]->PassParamToR3())
        {
            m_Imports[i]->FillParamStruct(&(m_pImpStructs[j++]));
            m_Imports[i]->InhibitParamToR3();
        }
    }
    m_pImpStructs[nImpCount].name = NULL;
}

void CRfcFunction::SetUpExports(void)
{
    int nExpCount = m_Exports.GetSize();
 
    if( m_bExportsInvalid )
    {    
        if( m_pExpStructs!=NULL ) delete[] m_pExpStructs;  
        m_pExpStructs 
           = (RFC_PARAMETER *)new BYTE[sizeof(RFC_PARAMETER)*(nExpCount+1)];
        if( m_pExpStructs == NULL )
        {
            RfcClassTrace.TraceTime();
            RfcClassTrace.Trace(CRFCFUNC_SETUPEXPORTS);
            RfcClassTrace.Trace(MEMORY);
            throw MEMORY;
        }
        m_bExportsInvalid=FALSE;
    }

    for( int i=0; i<nExpCount; i++ )
        m_Exports[i]->FillParamStruct(&(m_pExpStructs[i]));
    m_pExpStructs[nExpCount].name = NULL;

}

void CRfcFunction::SetUpTables(void)
{
    int nTabCount = m_Tables.GetSize();

    if( m_bTablesInvalid )
    {    
        if( m_pTabStructs!=NULL ) delete[] m_pTabStructs;  
        m_pTabStructs 
           = (RFC_TABLE *)new BYTE[sizeof(RFC_TABLE)*(nTabCount+1)];
        if( m_pTabStructs == NULL )
        {
            RfcClassTrace.TraceTime();
            RfcClassTrace.Trace(CRFCFUNC_SETUPTABLES);
            RfcClassTrace.Trace(MEMORY);
            throw MEMORY;
        }
        m_bTablesInvalid=FALSE;
    }
    
    for( int i=0; i<nTabCount; i++ )
        m_Tables[i]->FillTableStruct(&(m_pTabStructs[i]));
    m_pTabStructs[nTabCount].name = NULL; 
}

void CRfcFunction::ExtractTables(void)
{
    for( int i=0; i<m_Tables.GetSize(); i++ )
        m_Tables[i]->ExtractTableHandle(&(m_pTabStructs[i]));
}


void CRfcFunction::ConvertExports(void)
{
    for( int i=0; i<m_Exports.GetSize(); i++ )
    {
        // Conversion only needed for structure params
        if( m_Exports[i]->IsSimpleParam() == FALSE )
            m_Exports[i]->ConvertData();
    }
}


void CRfcFunction::CheckTables(void) const
{

    for( int i=0; i<m_Tables.GetSize(); i++ )
	{
        assert( m_pTabStructs[i].ithandle != ITAB_NULL );
		if (m_pTabStructs[i].ithandle == ITAB_NULL)
			throw CRFCFUNC_CHECKTAB_NULL_TAB_HANDLE;
	}
}


void CRfcFunction::SetFunctionName (CSTR name)
{ 
	assert(  m_rstrFuncName == (CSTR)NULL || m_rstrFuncName.GetLength()==0); 
	if (m_rstrFuncName == (CSTR)NULL || m_rstrFuncName.GetLength()==0)
		m_rstrFuncName=name; 
}


//  This function adds one program name to the list of the programs that the RFC
//  client or server application wants the RFC library automatically to start.
//  Input:  Program's name, a character string.
//  Output: No explicit output, but the class variable, the list of program names
//          has been modified.
//  Return: Nothing.
//  Exception:  Throws const char* for memory allocation exception.
void CRfcFunction::AddProgram(char *program_name)
{
	if (program_name == NULL) return;

	if (m_ProgramNames.GetLength() == 0)
	{
		m_ProgramNames = program_name;
	}
	else
	{
		m_ProgramNames += ";";
		m_ProgramNames += program_name;
	}
}


//  This function will start the programs in the program list (m_ProgramNames)
//  if *program_names=NULL or the programs listed in the program arguments program_names.
//  Input: Nothing or Program's name, a character string.
//  Return: Nothing.
//  Exception: Throws RFC_ERROR_INFO_EX for errors during RfcAllowStartProgram call.
//             Throws const char* for memory allocation exception.
void CRfcFunction::AllowStartPrograms(char *program_names)
{
	if (program_names != NULL)
	{
		CRfcString tempString(*program_names);
		AllowStartProgramsWithSeparator(tempString);
	}
	else if (m_ProgramNames != (CSTR)NULL)
	{
		AllowStartProgramsWithSeparator(m_ProgramNames);
	}
}


//  Helper function for StartPrograms.

void CRfcFunction::AllowStartProgramsWithSeparator(CRfcString & program_names)
{
	CRfcString remStr;
	CRfcString progNames;
	CRfcString progName;

	remStr = program_names;
	RFC_RC rc;
	int status=0;
	while (status == 0)
	{
		progNames = remStr;
		status = CRfcString::GetSubStr((char *) (CSTR)progNames, (char *) (CSTR)progName, (char *) (CSTR)remStr, ';');
		rc = ::RfcAllowStartProgram((char *) (CSTR)progName);
		if( rc != RFC_OK )
		{
			CRFC_ERROR_INFO errorInfo;
			::RfcLastErrorEx(&errorInfo);
			::RfcLastError(&errorInfo);
			throw errorInfo;
			return;
		}
	}
}
